### Satisfaction with Democracy: When Government by the People brings Electoral Losers and Winners Together
# Code Lucas Leemann and Isabelle Stadelmann-Steffen
# Validation: Link between national and subnational democratic satisfaction
# First version: December 2020
# Final version: April 2021

data <- read.dta("Data/Individual/2018_04_06_Aggregated_dataset_v2_stata12.dta", convert.dates = TRUE, convert.factors = TRUE,
         missing.type = FALSE,
         convert.underscore = FALSE, warn.missing.labels = TRUE)


#Data preparation

dataMEDW <- data%>%filter(COUNTRY == "Switzerland" | COUNTRY == "Germany")

#Democratic satisfaction national
dataMEDW$sf_national <- dataMEDW$Q1A_BEGINNING
dataMEDW$sf_national <- na_if(dataMEDW$sf_national, 99)

#Democratic satisfaction regional
dataMEDW$sf_regional <- dataMEDW$Q1B_BEGINNING
dataMEDW$sf_regional <- na_if(dataMEDW$sf_regional, 99)

satis<- dataMEDW%>% select(sf_regional,sf_national,REGIONID)
satis <- na.omit(satis)


#Correlations
func <- function(satis)
{
  return(data.frame(COR = cor(satis$sf_national, satis$sf_regional)))
}

corr <- ddply(satis, .(REGIONID), func)

#Figure A4, upper graph
corrPlotMEDW <- ggplot(corr, aes(x = REGIONID, y=COR)) + geom_point(shape = 3, size = 5) + ylim(0,1) +
  xlab("Subnational units") + ylab("Correlation coefficient R") + ggtitle("Data: Making Electoral Democracy Work")
corrPlotMEDW


#Regressions: Replicating individual level models with subnational democratic satisfaction
#########################################################

#Prepare data based on "2_Combine L1 and L2 data_USnew.R"

dataREG <- dataMEDW%>%select(sf_national, sf_regional,GEND, YOB, REGIONID, Q7, Q18, Q46B, Q47_1, SD2A, POSTSECONDARY, PSD3, COUNTRY)


dataREG$REGIONID[dataREG$REGIONID=="Z\xfcrich"] <- "Zurich"

# age
dataREG$age <- 2017-dataREG$YOB
dataREG$age[dataREG$age<0] <- NA
dataREG$age[dataREG$age>100] <- NA

# female
dataREG$female <- "male"
dataREG$female[dataREG$GEND=="Female"] <- "female"

# education
dataREG$edu<- "low edu"
dataREG$edu[dataREG$POSTSECONDARY=="Yes"]<- "postsec. edu"
table(dataREG$edu)

# employed? (full time) => not available

# participation (High values, low participation probability)
dataREG$participation <- as.numeric(dataREG$Q7)
dataREG$participation[dataREG$Q7=="Don't know"] <- NA
table(dataREG$participation)
dataREG$participation<- as.numeric(dataREG$participation)

dataREG <- dataREG%>%select(sf_national, sf_regional,COUNTRY, participation, edu, female, age, REGIONID, Q47_1)

dataREG <- na.omit(dataREG)

# read-in data on subnational parties
DE.parties <- read_excel("Data/Subnationale Exekutiven.xlsx", sheet="DE2")
CH.parties <- read_excel("Data/Subnationale Exekutiven.xlsx", sheet="CH")

dataREG$ID <- NA
dataREG$ID[dataREG$REGIONID=="Bavaria"] <- 8
dataREG$ID[dataREG$REGIONID=="Lower Saxony"] <- 16
dataREG$ID[dataREG$REGIONID=="Lucerne"] <- 3
dataREG$ID[dataREG$REGIONID=="Zurich"] <- 1

# Parties Germany
# parties -- 1: SPD	2:CDU	(CSU)	4:Greens	5:Linke	6:FDP	
dataREG$partiesDE <- 0
dataREG$partiesDE[dataREG$Q47_1=="Party 2"& dataREG$COUNTRY=="Germany"] <- 1
dataREG$partiesDE[dataREG$Q47_1=="Party 1"& dataREG$COUNTRY=="Germany"] <- 2
dataREG$partiesDE[dataREG$Q47_1=="Party 3"& dataREG$COUNTRY=="Germany"] <- 4
dataREG$partiesDE[dataREG$Q47_1=="Party 6"& dataREG$COUNTRY=="Germany"] <- 5
dataREG$partiesDE[dataREG$Q47_1=="Party 5"& dataREG$COUNTRY=="Germany"] <- 6

dataREG$winnerDE <- 0

www <- as.numeric(names(table(dataREG$ID)))

for (i in 1:length(dataREG$winnerDE)){
  cr <- dataREG$ID[i]
  pp <- as.numeric(DE.parties[as.numeric(as.character(cr)),])[-c(1:2)]
  parties.in.power <- which(pp==1)
  if (dataREG$partiesDE[i] %in% parties.in.power) dataREG$winnerDE[i] <- 1
}

#Parties Switzerland
# parties -- 1:FDP	2:CVP	3:SP	4:SVP	5:LP	6:CSP	7:BDP	8:GP	9:Lega	10: MCG	Übrige
dataREG$partiesCH
dataREG$partiesCH[dataREG$Q47_1=="Party 1"& dataREG$COUNTRY=="Switzerland"] <- 2
dataREG$partiesCH[dataREG$Q47_1=="Party 2"& dataREG$COUNTRY=="Switzerland"] <- 1
dataREG$partiesCH[dataREG$Q47_1=="Party 3"& dataREG$COUNTRY=="Switzerland"] <- 4
dataREG$partiesCH[dataREG$Q47_1=="Party 4"& dataREG$COUNTRY=="Switzerland"] <- 3
dataREG$partiesCH[dataREG$Q47_1=="Party 5"& dataREG$COUNTRY=="Switzerland"] <- 8
dataREG$partiesCH[dataREG$Q47_1=="Party 6"& dataREG$COUNTRY=="Switzerland"] <- 7
table(dataREG$partiesCH)

dataREG$winnerCH <- rep(0,dim(dataREG)[1])
for (i in 1:length(dataREG$winnerCH)){
  cr <- dataREG$ID[i]
  pp <- as.numeric(CH.parties[as.numeric(as.character(cr)),])[-c(1:2)]
  parties.in.power <- which(pp==1)
  if (dataREG$partiesCH[i] %in% parties.in.power) dataREG$winnerCH[i] <- 1
}

dataREG$winner <- dataREG$winnerCH + dataREG$winnerDE
table(dataREG$winner)
table(dataREG$winner, dataREG$COUNTRY)



#Regression as shown in Table 7 (left part, MEDW)
m1 <- lm(sf_national ~ winner + age + age*age + female + edu + participation + as.factor(REGIONID), data=dataREG)
m2 <- lm(sf_regional ~ winner + age + age*age + female + edu + participation + as.factor(REGIONID), data=dataREG)


screenreg(list(m1,m2))

#########################################################
# Second part: Analyses based on different GLES Landtagswahlen
# Data: For every subnational unit analysed, see the specific data link
# 
##################################################################


#Landtagswahlen Nordrhein-Westfalen 2017 (GLES)
#GLES (2017). Langfrist-Online-Tracking zur Landtagswahl in Nordrhein-Westfalen 2017 (GLES). 
#GESIS Datenarchiv, K?ln. ZA6820 Datenfile Version 1.0.0, https://doi.org/10.4232/1.12852.

data1 <- read.dta("ZA6820_v1-0-0.dta", convert.dates = TRUE, convert.factors = TRUE,
                 missing.type = FALSE,
                 convert.underscore = FALSE, warn.missing.labels = TRUE)

#Demokratiezufriedenheit national: l7b
#Demokratiezufriedenheit subnational: l7a


data1$sf_regional <- NA
data1$sf_regional[data1$l7a=="sehr zufrieden"]<- "Very satisfied"
data1$sf_regional[data1$l7a=="zufrieden"]<- "Satisfied"
data1$sf_regional[data1$l7a=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data1$sf_regional[data1$l7a=="unzufrieden"]<- "Unsatisfied"
data1$sf_regional[data1$l7a=="sehr unzufrieden"]<- "Very unsatisfied"
data1$sf_regional <- ordered(data1$sf_regional, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

data1$sf_national <- NA
data1$sf_national[data1$l7b=="sehr zufrieden"]<- "Very satisfied"
data1$sf_national[data1$l7b=="zufrieden"]<- "Satisfied"
data1$sf_national[data1$l7b=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data1$sf_national[data1$l7b=="unzufrieden"]<- "Unsatisfied"
data1$sf_national[data1$l7b=="sehr unzufrieden"]<- "Very unsatisfied"
data1$sf_national <- ordered(data1$sf_national, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

satis<- data1%>%select(sf_regional,sf_national)

data1$sf_national <- as.numeric(data1$sf_national)
data1$sf_regional <- as.numeric(data1$sf_regional)
satis2<- data1%>%select(sf_regional,sf_national)
satis2 <- na.omit(satis2)

c1 <- cor(satis2)
c2 <- cor.test(satis2$sf_regional, satis2$sf_national)

corr_NRW <- c("NRW", c1[1,2], c2$p.value)



#Landtagswahlen Baden-W?rttemberg 2016 (GLES)
#GLES (2016). Langfrist-Online-Tracking zur Landtagswahl in Baden-W?rttemberg 2016 (GLES). 
#GESIS Datenarchiv, K?ln. ZA5741 Datenfile Version 1.0.0, https://doi.org/10.4232/1.12531.

rm(satis, satis2, c1, c2)
data2 <- read.dta("ZA5741_v1-0-0.dta", convert.dates = TRUE, convert.factors = TRUE,
                 missing.type = FALSE,
                 convert.underscore = FALSE, warn.missing.labels = TRUE)

data2$sf_regional <- NA
data2$sf_regional[data2$l7a=="sehr zufrieden"]<- "Very satisfied"
data2$sf_regional[data2$l7a=="zufrieden"]<- "Satisfied"
data2$sf_regional[data2$l7a=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data2$sf_regional[data2$l7a=="unzufrieden"]<- "Unsatisfied"
data2$sf_regional[data2$l7a=="sehr unzufrieden"]<- "Very unsatisfied"
data2$sf_regional <- ordered(data2$sf_regional, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

data2$sf_national <- NA
data2$sf_national[data2$l7b=="sehr zufrieden"]<- "Very satisfied"
data2$sf_national[data2$l7b=="zufrieden"]<- "Satisfied"
data2$sf_national[data2$l7b=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data2$sf_national[data2$l7b=="unzufrieden"]<- "Unsatisfied"
data2$sf_national[data2$l7b=="sehr unzufrieden"]<- "Very unsatisfied"
data2$sf_national <- ordered(data2$sf_national, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

satis <- data2%>%select(sf_regional, sf_national)
satis <- subset(data2, select = c("sf_regional", "sf_national"))

data2$sf_national <- as.numeric(data2$sf_national)
data2$sf_regional <- as.numeric(data2$sf_regional)
satis2<- data2%>%select(sf_regional,sf_national)
satis2 <- na.omit(satis2)
c1 <- cor(satis2)
c2 <- cor.test(satis2$sf_regional, satis2$sf_national)

corr_BW <- c("BW", c1[1,2], c2$p.value)

# Landtagswahlen Mecklenburg-Vorpommern 2016
#GLES (2016). Langfrist-Online-Tracking zur Landtagswahl in Mecklenburg-Vorpommern 2016 (GLES). 
#GESIS Datenarchiv, K?ln. ZA5744 Datenfile Version 1.0.0, https://doi.org/10.4232/1.12674.

rm(satis, satis2, c1, c2)
data3 <- read.dta("ZA5744_v1-0-0.dta", convert.dates = TRUE, convert.factors = TRUE,
                  missing.type = FALSE,
                  convert.underscore = FALSE, warn.missing.labels = TRUE)

data3$sf_regional <- NA
data3$sf_regional[data3$l7a=="sehr zufrieden"]<- "Very satisfied"
data3$sf_regional[data3$l7a=="zufrieden"]<- "Satisfied"
data3$sf_regional[data3$l7a=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data3$sf_regional[data3$l7a=="unzufrieden"]<- "Unsatisfied"
data3$sf_regional[data3$l7a=="sehr unzufrieden"]<- "Very unsatisfied"
data3$sf_regional <- ordered(data3$sf_regional, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

data3$sf_national <- NA
data3$sf_national[data3$l7b=="sehr zufrieden"]<- "Very satisfied"
data3$sf_national[data3$l7b=="zufrieden"]<- "Satisfied"
data3$sf_national[data3$l7b=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data3$sf_national[data3$l7b=="unzufrieden"]<- "Unsatisfied"
data3$sf_national[data3$l7b=="sehr unzufrieden"]<- "Very unsatisfied"
data3$sf_national <- ordered(data3$sf_national, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

satis <- data3%>%select(sf_regional, sf_national)
satis <- subset(data3, select = c("sf_regional", "sf_national"))

data3$sf_national <- as.numeric(data3$sf_national)
data3$sf_regional <- as.numeric(data3$sf_regional)
satis2<- data3%>%select(sf_regional,sf_national)
satis2 <- na.omit(satis2)
c1 <- cor(satis2)
c2 <- cor.test(satis2$sf_regional, satis2$sf_national)

corr_MVP <- c("MVP", c1[1,2], c2$p.value)



#Landtagswahlen Rheinland_pfalz 2016 (GLES)
#GLES (2016). Langfrist-Online-Tracking zur Landtagswahl in Rheinland-Pfalz 2016 (GLES). 
#GESIS Datenarchiv, K?ln. ZA5743 Datenfile Version 1.0.0, https://doi.org/10.4232/1.12533.


rm(satis, satis2, c1, c2)
data4 <- read.dta("ZA5743_v1-0-0.dta", convert.dates = TRUE, convert.factors = TRUE,
                 missing.type = FALSE,
                 convert.underscore = FALSE, warn.missing.labels = TRUE)

data4$sf_regional <- NA
data4$sf_regional[data4$l7a=="sehr zufrieden"]<- "Very satisfied"
data4$sf_regional[data4$l7a=="zufrieden"]<- "Satisfied"
data4$sf_regional[data4$l7a=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data4$sf_regional[data4$l7a=="unzufrieden"]<- "Unsatisfied"
data4$sf_regional[data4$l7a=="sehr unzufrieden"]<- "Very unsatisfied"
data4$sf_regional <- ordered(data4$sf_regional, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

data4$sf_national <- NA
data4$sf_national[data4$l7b=="sehr zufrieden"]<- "Very satisfied"
data4$sf_national[data4$l7b=="zufrieden"]<- "Satisfied"
data4$sf_national[data4$l7b=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data4$sf_national[data4$l7b=="unzufrieden"]<- "Unsatisfied"
data4$sf_national[data4$l7b=="sehr unzufrieden"]<- "Very unsatisfied"
data4$sf_national <- ordered(data4$sf_national, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

satis <- subset(data4, select = c("sf_regional", "sf_national"))

data4$sf_national <- as.numeric(data4$sf_national)
data4$sf_regional <- as.numeric(data4$sf_regional)
satis2<- data4%>%select(sf_regional,sf_national)
satis2 <- na.omit(satis2)
c1 <- cor(satis2)
c2 <- cor.test(satis2$sf_regional, satis2$sf_national)

corr_RP <- c("RP", c1[1,2], c2$p.value)


#Landtagswahlen Sachsen-Anhalt 2016 (GLES)
#GLES (2016). Langfrist-Online-Tracking zur Landtagswahl in Sachsen-Anhalt 2016 (GLES). 
#GESIS Datenarchiv, K?ln. ZA5742 Datenfile Version 1.0.0, https://doi.org/10.4232/1.12532.

rm(satis, satis2, c1, c2)
data5 <- read.dta("ZA5742_v1-0-0.dta", convert.dates = TRUE, convert.factors = TRUE,
                 missing.type = FALSE,
                 convert.underscore = FALSE, warn.missing.labels = TRUE)

data5$sf_regional <- NA
data5$sf_regional[data5$l7a=="sehr zufrieden"]<- "Very satisfied"
data5$sf_regional[data5$l7a=="zufrieden"]<- "Satisfied"
data5$sf_regional[data5$l7a=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data5$sf_regional[data5$l7a=="unzufrieden"]<- "Unsatisfied"
data5$sf_regional[data5$l7a=="sehr unzufrieden"]<- "Very unsatisfied"
data5$sf_regional <- ordered(data5$sf_regional, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

data5$sf_national <- NA
data5$sf_national[data5$l7b=="sehr zufrieden"]<- "Very satisfied"
data5$sf_national[data5$l7b=="zufrieden"]<- "Satisfied"
data5$sf_national[data5$l7b=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data5$sf_national[data5$l7b=="unzufrieden"]<- "Unsatisfied"
data5$sf_national[data5$l7b=="sehr unzufrieden"]<- "Very unsatisfied"
data5$sf_national <- ordered(data5$sf_national, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

satis <- subset(data5, select = c("sf_regional", "sf_national"))

data5$sf_national <- as.numeric(data5$sf_national)
data5$sf_regional <- as.numeric(data5$sf_regional)
satis2<- data5%>%select(sf_regional,sf_national)
satis2 <- na.omit(satis2)
c1 <- cor(satis2)
c2 <- cor.test(satis2$sf_regional, satis2$sf_national)

corr_SA <- c("SA", c1[1,2], c2$p.value)




#Landtagswahlen Scheswig-Holstein 2017 (GLES)
#GLES (2017). Langfrist-Online-Tracking zur Landtagswahl in Schleswig-Holstein 2017 (GLES). 
#GESIS Datenarchiv, K?ln. ZA6819 Datenfile Version 1.0.0, https://doi.org/10.4232/1.12851.

rm(satis, satis2, c1, c2)
data6 <- read.dta("ZA6819_v1-0-0.dta", convert.dates = TRUE, convert.factors = TRUE,
                 missing.type = FALSE,
                 convert.underscore = FALSE, warn.missing.labels = TRUE)

data6$sf_regional <- NA
data6$sf_regional[data6$l7a=="sehr zufrieden"]<- "Very satisfied"
data6$sf_regional[data6$l7a=="zufrieden"]<- "Satisfied"
data6$sf_regional[data6$l7a=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data6$sf_regional[data6$l7a=="unzufrieden"]<- "Unsatisfied"
data6$sf_regional[data6$l7a=="sehr unzufrieden"]<- "Very unsatisfied"
data6$sf_regional <- ordered(data6$sf_regional, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

data6$sf_national <- NA
data6$sf_national[data6$l7b=="sehr zufrieden"]<- "Very satisfied"
data6$sf_national[data6$l7b=="zufrieden"]<- "Satisfied"
data6$sf_national[data6$l7b=="teils zufrieden, teils unzufrieden"]<- "Partly satisfied, partly unsatisfied"
data6$sf_national[data6$l7b=="unzufrieden"]<- "Unsatisfied"
data6$sf_national[data6$l7b=="sehr unzufrieden"]<- "Very unsatisfied"
data6$sf_national <- ordered(data6$sf_national, levels = c("Very unsatisfied", "Unsatisfied", "Partly satisfied, partly unsatisfied", "Satisfied", "Very satisfied"))

satis <- subset(data6, select = c("sf_regional", "sf_national"))

data6$sf_national <- as.numeric(data6$sf_national)
data6$sf_regional <- as.numeric(data6$sf_regional)
satis2<- data6%>%select(sf_regional,sf_national)
satis2 <- na.omit(satis2)
c1 <- cor(satis2)
c2 <- cor.test(satis2$sf_regional, satis2$sf_national)

corr_SH <- c("SH", c1[1,2], c2$p.value)

rm(satis, satis2, c1, c2)


#Plotting and analyses
##########################################################

#Correlation plot (Figure A4, lower graph)
corr <- rbind(corr_BW, corr_MVP, corr_NRW, corr_RP, corr_SA, corr_SH)
corr <- as_tibble(corr)
corr$V2 <- as.numeric(corr$V2)
corr$V3 <- as.numeric(corr$V3)
corr %>% mutate(V2=round(V2,2))

corrPlotDE <- ggplot(corr, aes(x = V1, y=V2)) + geom_point(shape = 3, size = 5) + ylim(0,1) +
  xlab("Subnational units Germany") + ylab("Correlation coefficient R") + ggtitle("Data: GLES, subnational elections")
corrPlotDE




#Replicating individual level models with subnational democratic satisfaction
#Germany
#########################################################

#Prepare data based on "2_Combine L1 and L2 data_USnew.R"

data1a <- data1%>%select(survey, sf_national, sf_regional, l2, l1, l349a:l349n, l181, l70, l146a)
data2a <- data2%>%select(survey, sf_national, sf_regional, l2, l1, l349a:l349n, l181, l70, l146a)
data3a <- data3%>%select(survey, sf_national, sf_regional, l2, l1, l349a:l349n, l181, l70, l146a)
data4a <- data4%>%select(survey, sf_national, sf_regional, l2, l1, l349a:l349n, l181, l70, l146a)
data5a <- data5%>%select(survey, sf_national, sf_regional, l2, l1, l349a:l349n, l181, l70, l146a)
data6a <- data6%>%select(survey, sf_national, sf_regional, l2, l1, l349a:l349n, l181, l70, l146a)

dataREGde <- rbind(data1a, data2a, data3a, data4a, data5a, data6a)

dataREGde$bula <- NA
dataREGde$bula[dataREGde$survey=="Landtagswahl Nordrhein-Westfalen 2017"] <- 5 #NWF
dataREGde$bula[dataREGde$survey=="Landtagswahl Baden-Wuerttemberg 2016"] <- 8 #NWF
dataREGde$bula[dataREGde$survey=="Landtagswahl Mecklenburg-Vorpommern 2016"] <- 13 #NWF
dataREGde$bula[dataREGde$survey=="Landtagswahl Rheinland-Pfalz 2016"] <- 7 #NWF
dataREGde$bula[dataREGde$survey=="Landtagswahl Sachsen-Anhalt 2016"] <- 15 #NWF
dataREGde$bula[dataREGde$survey=="Landtagswahl Schleswig-Holstein 2017"] <- 1 #NWF
table(dataREGde$bula)


# age
dataREGde$age <- 2017-dataREGde$l2
dataREGde$age[dataREGde$age<0] <- NA
dataREGde$age[dataREGde$age>100] <- NA
summary(dataREGde$age)

# female
dataREGde$female <- "male"
dataREGde$female[dataREGde$l1=="weiblich"] <- "female"
table(dataREGde$female)

# education
dataREGde$edu<- NA
# education nach ISCED
dataREGde$edu[dataREGde$l3==9|dataREGde$l349l=="genannt"|dataREGde$l349m=="genannt"] <- 1 
dataREGde$edu[dataREGde$l349a=="genannt"|dataREGde$l349b=="genannt"|dataREGde$l349e=="genannt"] <- 1
dataREGde$edu[dataREGde$l349c=="genannt"|dataREGde$l349d=="genannt"|dataREGde$l349f=="genannt"|dataREGde$l349g=="genannt"|dataREGde$l349k=="genannt"] <- 2
dataREGde$edu[dataREGde$l349h=="genannt"|dataREGde$l349i=="genannt"|dataREGde$l349j=="genannt"|dataREGde$l349g=="genannt"] <- 3
dataREGde$edu <- as.factor(dataREGde$edu)
table(dataREGde$edu)


# employed? (full time)
dataREGde$employed <- NA

dataREGde$employed[dataREGde$l181=="Vollzeit berufstaetig (mehr als 30 Stunden/Woche)"] <- "Fulltime employed"
dataREGde$employed[dataREGde$l181=="zurzeit arbeitslos"] <- "Unemployed/disabled"
dataREGde$employed[dataREGde$l181=="Pensionaer/in, Rentner/in (frueher berufstaetig)"] <- "Retired"
dataREGde$employed[dataREGde$l181=="Student/in"|dataREGde$l181=="Schueler/in"||dataREGde$l181=="Lehrling/Azubi"|dataREGde$l181=="in Mutterschutz, Elternzeit"|dataREGde$l181=="nicht berufstaetig (Hausfrau/Hausmann)"|dataREGde$l181=="Bundesfreiwilligendienst, Freiwilliges Soziales Jahr (FSJ)"] <- "Student/homemaker"
dataREGde$employed[dataREGde$l181=="Teilzeit berufstaetig (bis 30 Stunden/Woche)"] <- "Parttime employed"
table(dataREGde$employed)
# 1: working full time, 2: unempl or disabled, 3: retired, 4: student/homemaker, 5: part-time

# participation
dataREGde$participation <- 0
dataREGde$participation[dataREGde$l70=="bestimmt zur Wahl gehen"|dataREGde$l70=="Briefwahl"] <- 1
dataREGde$participation[dataREGde$l70=="keine Angabe"|dataREGde$l70=="weiss nicht"] <- NA
table(dataREGde$participation)
#only 351 indicate to not be sure and not have voted by postal mail...



# read-in data on subnational parties

DE.parties <- read_excel("Data/Subnationale Exekutiven.xlsx", sheet="DE")

# parties -- 1: SPD	2:CDU	(CSU)	4:Greens	5:Linke	6:FDP	
dataREGde$parties <- 0
dataREGde$parties[dataREGde$l146a=="SPD"] <- 1
dataREGde$parties[dataREGde$l146a=="CDU/CSU"|dataREGde$l146a=="CSU"|dataREGde$l146a=="CDU"] <- 2
dataREGde$parties[dataREGde$l146a=="GRUENE"] <- 4
dataREGde$parties[dataREGde$l146a=="DIE LINKE"] <- 5
dataREGde$parties[dataREGde$l146a=="FDP"] <- 6
dataREGde$parties[dataREGde$l146a=="FDP/DVP"] <- 6

table(dataREGde$parties)

dataREGde$winner <- rep(0,dim(dataREGde)[1])

www <- as.numeric(names(table(dataREGde$bula)))

for (i in 1:length(dataREGde$winner)){
  cr <- dataREGde$bula[i]
  pp <- as.numeric(DE.parties[as.numeric(as.character(cr)),])[-c(1:2)]
  parties.in.power <- which(pp==1)
  if (dataREGde$parties[i] %in% parties.in.power) dataREGde$winner[i] <- 1
}

table(dataREGde$winner)


#Regression
m3 <- lm(sf_national ~ winner + age + age*age + female + edu + employed + as.factor(bula), data=dataREGde)
m4 <- lm(sf_regional ~ winner + age + age*age + female + edu + employed + as.factor(bula), data=dataREGde)
library(stargazer)
stargazer(m3, m4, 
          type = 'text')


#Combining models and figures from data sets
####################################################

#Regression models (Table A7)
screenreg(list(m1, m2, m3, m4),
          file="Output/Tables and Plots/Table_A7.txt")


###Figure A4 final
plotsCorr <- ggarrange(corrPlotMEDW, corrPlotDE, 
                       ncol = 1, nrow = 2)
plotsCorr

ggsave("Output/Tables and Plots/Fig A4.png",
       dpi=300,
       units=c("in"),
       width=10, height=6)

